﻿using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using Inet.Viewer.Data;

namespace Inet.Viewer.WinForms.Prompt
{
    /// <summary>
    /// panel for a prompt field which allows for multiple prompt values to be chosen and
    /// added to a list
    /// </summary>
    public partial class MultiPromptField : PromptControl
    {
        internal PromptControl pnlSingle;
        internal PromptControl pnlRange;
        private IButtonControl oldAcceptButton;

        /// <summary>
        /// simple constructor with the necessary dependencies
        /// </summary>
        /// <param name="p">prompt this panel is based on</param>
        /// <param name="singleControl">the single prompt panel part (can be null if no single values are allowed)</param>
        /// <param name="rangeControl">the range prompt panel part (can be null if no range values are allowed)</param>
        public MultiPromptField(Data.PromptData p, PromptControl singleControl, PromptControl rangeControl)
        {
            InitializeComponent();
            this.PromptData = p;
            this.pnlSingle = singleControl;
            this.pnlRange = rangeControl;
            if (singleControl != null)
            {
                btnAddSingle.Enabled = singleControl.ValidatePrompt();
                singleControl.ValueChanged += (s, e) => { btnAddSingle.Enabled = singleControl.ValidatePrompt(); };
                panelSingle.Controls.Add(singleControl);
                groupBoxSingle.Size = new Size(panelSingle.Size.Width, singleControl.Height + btnAddSingle.Height + panelSingle.Location.Y + 4);
            }
            else
            {
                groupBoxSingle.Visible = false;
            }

            if (rangeControl != null)
            {
                btnAddRange.Enabled = rangeControl.ValidatePrompt();
                rangeControl.ValueChanged += (s, e) => { btnAddRange.Enabled = rangeControl.ValidatePrompt(); };
                panelRange.Controls.Add(rangeControl);
                groupBoxRange.Size = new Size(panelRange.Size.Width, rangeControl.Height + btnAddRange.Height + panelRange.Location.Y + 4);
            }
            else
            {
                groupBoxRange.Visible = false;
            }

            this.btnDeleteEntry.Enabled = false;
            object v = p.Values;
            if (v is MultiPromptValue)
            {
                MultiPromptValue chosenValue = v as MultiPromptValue;
                ArrayList values = chosenValue.Values;
                this.listBox.Items.AddRange(values.ToArray());
            }
        }
 
        /// <summary>
        /// called to add a single element to the list
        /// </summary>
        /// <param name="sender">button which led to the event</param>
        /// <param name="e">eventargs of the action</param>
        private void button1_Click(object sender, EventArgs e)
        {
            if (pnlSingle.ValidatePrompt())
            {
                PromptValue value = pnlSingle.Value;
                if (value != null)
                {
                    this.listBox.Items.Add(value);
                }
            }
        }

        /// <summary>
        /// called to add a range element to the list
        /// </summary>
        /// <param name="sender">button which led to the event</param>
        /// <param name="e">eventargs of the action</param>
        private void button2_Click(object sender, EventArgs e)
        {
            if (pnlRange.ValidatePrompt())
            {
                PromptValue value = pnlRange.Value;
                if (value != null)
                {
                    this.listBox.Items.Add(value);
                }
            }
        }

        /// <summary>
        /// called to remove an element from the list
        /// </summary>
        /// <param name="sender">button which led to the event</param>
        /// <param name="e">eventargs of the action</param>
        private void button1_Click_1(object sender, EventArgs e)
        {
            if (this.listBox.SelectedIndex >= 0)
            {
                this.listBox.Items.RemoveAt(this.listBox.SelectedIndex);
                if (this.listBox.Items.Count == 0)
                {
                    this.btnDeleteEntry.Enabled = this.listBox.SelectedIndex >= 0;
                } 
            }
        }

        /// <summary>
        /// the MultiPromptValue chosen for this prompt field panel
        /// </summary>
        internal override PromptValue Value
        {
            get
            {
                MultiPromptValue result = new MultiPromptValue(this.PromptData.Type);
                result.Values.AddRange(this.listBox.Items);
                return result;
            }
        }
 
        /// <summary>
        /// when the index of the chosen element in the list of chosen elements is changed
        /// </summary>
        /// <param name="sender">control responsible for the action</param>
        /// <param name="e">eventargs of the action</param>
        private void listBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.btnDeleteEntry.Enabled = this.listBox.SelectedIndex >= 0;
        }

        internal override bool ValidatePrompt()
        {
            return true;
        }

        private void panelSingle_Enter(object sender, EventArgs e)
        {
            Form form = FindForm();
            
            if (form != null) {
                oldAcceptButton = form.AcceptButton;
                form.AcceptButton = btnAddSingle;
            }
        }

        private void panelRange_Enter(object sender, EventArgs e)
        {
            Form form = FindForm();
            if (form != null)
            {
                oldAcceptButton = form.AcceptButton;
                form.AcceptButton = btnAddRange;
            }
        }

        private void panel_Leave(object sender, EventArgs e)
        {
            Form form = FindForm();
            if (form != null)
            {
                form.AcceptButton = oldAcceptButton;
            }
        }
    }
}
